#ifndef _SOCKETSERVER_CPP
#define _SOCKETSERVER_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#define WIN32_LEAN_AND_MEAN
#define _WIN32_WINNT 0x0500

#include <Windows.H>
#include <WinSock2.H>
#include <WinSock.H>
#include <Stdio.H>
#include <Stdlib.H>
#include <IOStream>

#include "SockServer.H"
#include "../Source/Entry.H"
#include "../Source/Routines.H"

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//Non-standard definitions (Begin)

SocketServer gServer;

//Non-standard definitions (End)
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients DropBadPacketData flag.
*/
void SocketServer::DropBadPacketData(int iClient, bool bDropBad)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::DropBadPacketData.\n");
    #endif

    bcDropBadPackets[iClient] = bDropBad;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients bcRawDataMode flag.
*/
void SocketServer::SetRawStreamMode(int iClient, bool bRawMode)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SetRawStreamMode.\n");
    #endif

    bcRawDataMode[iClient] = bRawMode;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set the next client id.
*/
void SocketServer::SetNextClientID(int iClientID)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SetNextClientID.\n");
    #endif

	icNextClientID = iClientID;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to get the next client id.
*/
int SocketServer::GetNextClientID(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::GetNextClientID.\n");
    #endif

	return icNextClientID;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to get the number of currently connected clients.
*/
int SocketServer::GetCurrentClients(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::GetCurrentClients.\n");
    #endif

	return icCurrentClients;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function returns TRUE if the client has data in there receive buffer.

	NOTE: A TRUE return value does not mean that the data is a full packet,
	therfore a call to GetClientData or GetNextClientData may not return data.
*/
bool SocketServer::IsClientDataWaiting(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::IsClientDataWaiting.\n");
    #endif

    // Is data waiting to be received
    return (icRecvBufSz[iClient] > 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function returns TRUE if the client has data waiting to be sent.
*/
bool SocketServer::IsClientDataPending(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::IsClientDataPending.\n");
    #endif

    // Is data waiting to be sent?
    return (icSendBufSz[iClient] > 0);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will return TRUE when the client has a empty send buffer.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
*/
bool SocketServer::WaitOnClientDataToBeSent(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::WaitonClientDataToBeSent.\n");
    #endif

    bool bRetVal = false;

    while((bRetVal = IsClientDataPending(iClient)) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

    return !bRetVal;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will return TRUE when the client has data in there receive buffer.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
	
	NOTE: A TRUE return value does not mean that the data is a full packet,
	therfore a call to GetClientData or GetNextClientData may not return data.
*/
bool SocketServer::WaitOnClientDataToBeRecvd(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::WaitonClientDataToBeRecvd.\n");
    #endif

    bool bRetVal = false;

    while(!(bRetVal = IsClientDataWaiting(iClient)) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

    return bRetVal;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function will return TRUE when the client has data in there receive buffer
	and the data was a full packet.

	Returns FALSE if the client has been disconnected, is pending a disconnect or the
	data received is not a full packet.
*/

bool SocketServer::GetNextClientData(int iClient, char *outsBuf, int *outiSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::GetNextClientData.\n");
    #endif

    while(!IsClientDataWaiting(iClient) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

    return GetClientData(iClient, outsBuf, outiSize);
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to retrieve a clients data from there receive buffer.

	This function returns TRUE if the clients has data to be received,
	the the CriticalSection is open, the data is a full (and valid) packet.
	Otherwise the function returns FALSE.
*/
bool SocketServer::GetClientData(int iClient, char *outsBuf, int *outiSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::GetClientData.\n");
    #endif

    if(IsClientDataWaiting(iClient))
    {
        if(bcRawDataMode[iClient])
        {
            if(!TryEnterCriticalSection(&CriticalSection1))
                return false;

            if(bcConnected[iClient] && !bcDisconnect[iClient])
            {
                memcpy(outsBuf, scRecvBuf[iClient], icRecvBufSz[iClient]);
                *outiSize = icRecvBufSz[iClient];
            }
            else return false;

            LeaveCriticalSection(&CriticalSection1);

            icRecvBufSz[iClient] = 0;
        }
        else{
			if(!TryEnterCriticalSection(&CriticalSection1))
                return false;

			int iBreakPackedResult = 1000;

            if(bcConnected[iClient] && !bcDisconnect[iClient])
            {
				if( (iBreakPackedResult = BreakClassPacket(iClient, outsBuf, outiSize)) == 4)
				{
					LeaveCriticalSection(&CriticalSection1);
					return false;
				}
            }
			else{
				LeaveCriticalSection(&CriticalSection1);
				return false;
			}

            LeaveCriticalSection(&CriticalSection1);

            if(*outiSize < 0 && !bcDropBadPackets[iClient])
            {
				WriteLog(icClientID[iClient], "SocketServer::GetClientData : BreakClassPacket Failed.");
                bcDisconnect[iClient] = true;
                *outiSize = 0;
                return false;
            }
            else if(*outiSize < 0 && bcDropBadPackets[iClient]){
                icRecvBufSz[iClient] = 0;
                *outiSize = 0;
                return false;
            }

            if(iBreakPackedResult == 3)
			{
                icRecvBufSz[iClient] = 0;
			}
        }
        return true;
    }
    return false;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is called for each connecting client just
	before the client's bcConnected flag is set to TRUE.

	If this function returns FALSE the client wll not be accepted or connected.
*/
bool SocketServer::OnAcceptConnectClient(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::OnAcceptConnectClient.\n");
    #endif

    if((scRecvBuf[iClient] = (char *) calloc(sizeof(char), RECEIVEBUFSZ + 1)) == NULL)
    {
        return false;
    }

    if((scSendBuf[iClient] = (char *) calloc(sizeof(char), MAXSENDBUFSZ + 1)) == NULL)
    {
        return false;
    }

    icRecvOffset[iClient] = 0;
    icRecvBufSz[iClient] = 0;
    icSendBufSz[iClient] = 0;

	return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is called for each disconnecting client just
	after the client's bcConnected flag is set to FALSE.

	This functions return value has no meaning.

	NOTE: This function should not be called outside of the DisconnectClient function.
*/
bool SocketServer::OnDisconnectClient(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::OnDisconnectClient.\n");
    #endif

	WriteLog(icClientID[iClient], "Disconnected.");

	free(scRecvBuf[iClient]);
    free(scSendBuf[iClient]);

    icRecvBufSz[iClient] = 0;
    icSendBufSz[iClient] = 0;

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is called to disconnect a client.

	This functions return value has no meaning.

	NOTE: This function should not be called outside of the TCPHandler_Thread_Function.
*/
bool SocketServer::DisconnectClient(int iClient)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::DisconnectClient.\n");
    #endif

    if(!TryEnterCriticalSection(&CriticalSection1))
        return false;

    bcConnected[iClient] = false;
    bcDisconnect[iClient] = false;

    WaitOnClientThreadToExit(iClient);

    icCurrentClients--;

    WSASendDisconnect(sckcSocket[iClient], NULL);
	shutdown(sckcSocket[iClient], SD_BOTH);
	closesocket(sckcSocket[iClient]);

    bool bRetVal = OnDisconnectClient(iClient);

    LeaveCriticalSection(&CriticalSection1);

    return bRetVal;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function searched the bcConnected[] array to find a free client slot.

	If this function returns -1 the client wll not be accepted or connected.
*/
int SocketServer::SelectFreeSocket(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SelectFreeSocket.\n");
    #endif

	int iClient = 0;

	while(iClient < icMaxClients)
	{
		if( bcConnected[iClient] == false && bcDisconnect[iClient] == false)
            return iClient;

		iClient++;
	}

	return -1;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Initialize all of the class variables.
*/
bool SocketServer::Initialize(int iMaxClients)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::Initialize.\n");
    #endif

    if(!Start_WindowsSocket())
    {
		WriteSysLog("SocketServer::Start_WindowsSocket : Failed.");
        return false;
	}

    icMaxClients = iMaxClients;
    icCurrentClients = 0;
    icNextClientID = 0;

    if((scSendBuf = (char **) calloc(sizeof(char *), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((scRecvBuf = (char **) calloc(sizeof(char *), iMaxClients + 1)) == NULL)
    {
        return false;
    }

	if((icSendBufSz = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((icRecvBufSz = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((bcConnected = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((bcDisconnect = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((sckcSocket = (SOCKET *) calloc(sizeof(SOCKET), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((icClientID = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((dwcClient_Thread_ID = (DWORD *) calloc(sizeof(DWORD), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((hcClient_Thread_Handle = (HANDLE *) calloc(sizeof(HANDLE), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((icRecvOffset = (int *) calloc(sizeof(int), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((bcRawDataMode = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    if((bcDropBadPackets = (bool *) calloc(sizeof(bool), iMaxClients + 1)) == NULL)
    {
        return false;
    }

    InitializeCriticalSection(&CriticalSection1);

    int iClient = 0;
    while(iClient < iMaxClients)
	{
		bcConnected[iClient] = false;
        bcDisconnect[iClient] = false;
		iClient++;
	}

    IsListeningServer = false;

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Destroy all of the class variables.
*/
bool SocketServer::UnInitialize(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::UnInitialize.\n");
    #endif

	free(bcDropBadPackets);
	free(bcConnected);
	free(bcDisconnect);
	free(bcRawDataMode);
	free(dwcClient_Thread_ID);
	free(hcClient_Thread_Handle);
	free(icClientID);
	free(icRecvBufSz);
	free(icRecvOffset);
	free(icSendBufSz);
	free(sckcSocket);
	free(scRecvBuf);
	free(scSendBuf);

    DeleteCriticalSection(&CriticalSection1);

	if(!Stop_WindowsSocket())
    {
		WriteSysLog("SocketServer::UnInitialize : Stop_WindowsSocket Failed.");
        return false;
	}

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function sets up a socket to listen for incomming connections.
	This function should not be called outside of the Start() function.
*/
bool SocketServer::Open_ListenSocket(int iListenPort)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::Open_ListenSocket.\n");
    #endif

	SOCKADDR_IN LocalListenSocketAddress; // Socket address of local server

	// ----- Fill In The Address Structure For Local Server
	LocalListenSocketAddress.sin_family      = AF_INET;		       // Address Family
	LocalListenSocketAddress.sin_addr.s_addr = INADDR_ANY;         // Let socket Library Assign Address
	LocalListenSocketAddress.sin_port        = htons(iListenPort); // Port Number

	// ----- Create A TCP/IP Stream Socket To "Listen" with -----
	if((sckcListenSocket = socket(AF_INET, SOCK_STREAM, IPPROTO_TCP)) == INVALID_SOCKET)
	{
		WriteSysLog("SocketServer::Open_ListenSocket : socket failed.");
		return false;
	}

	// -------- Bind The Name To The Socket
	if(bind(sckcListenSocket, (SOCKADDR*)&LocalListenSocketAddress, sizeof(struct sockaddr)) == SOCKET_ERROR)
	{
		WriteSysLog("SocketServer::Open_ListenSocket : bind failed.");
		closesocket(sckcListenSocket);
		return false;
	}

	// -------- Set The Socket To Listen
	if(listen(sckcListenSocket, SOMAXCONN) == SOCKET_ERROR)
	{
		WriteSysLog("SocketServer::Open_ListenSocket : listen failed.");
		closesocket(sckcListenSocket);
		return false;
	}

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function closes down the listening socket.
*/
bool SocketServer::Close_ListenSocket(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::Close_ListenSocket.\n");
    #endif

    if(closesocket(sckcListenSocket) == SOCKET_ERROR)
    {
        return false;
    }

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Start the client or server.

	If the iListenPort is non-zero a listening port will be opened to
	accept incomming connections.
*/
bool SocketServer::Start(int iListenPort)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::Start.\n");
    #endif

    if(iListenPort > 0)
    {
        if(!Open_ListenSocket(iListenPort))
        {
			WriteSysLog("SocketServer::Start : Open_ListenSocket failed.");
            return false;
        }
        IsListeningServer = true;
    }

    if(!Start_TCPHandler())
    {
		WriteSysLog("SocketServer::Start : Start_TCPHandler failed.");
        return false;
    }

    return true;
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to Stop the server or client. If there is a listening socket
	it will be shutdown and closed.
*/
bool SocketServer::Stop(void)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::Stop.\n");
    #endif

    if(!Stop_TCPHandler())
    {
    	WriteSysLog("SocketServer::Stop : Stop_TCPHandler failed.");
        return false;
    }

    if(IsListeningServer)
    {
        if(!Close_ListenSocket())
        {
			WriteSysLog("SocketServer::Stop : Close_ListenSocket failed.");
            return false;
        }
        IsListeningServer = false;
    }

    return true;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to actually send a clients data. It should never be called manually
	outside of DoSend or SendData.
*/
int SocketServer::SendDataEx(int iClient, char *sData, int iSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SendDataEx.\n");
    #endif

    int iRetVal = 0;

    if(bcRawDataMode[iClient])
    {
    	iRetVal = send(sckcSocket[iClient], sData, iSize, 0);
        return iRetVal;
    }
    else {

        char sSendData[MAXSENDBUFSZ];
        int iSendLen = MakeClassPacket(sSendData, sData, iSize);
    	iRetVal = send(sckcSocket[iClient], sSendData, iSendLen, 0);
        return iRetVal;
    }

	WriteSysLog("SocketServer::SendDataEx : Logic failure.");

    return -1;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to actually send a clients data. It should never be called manually.
*/
int SocketServer::SendData(int iClient, char *sData)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SendData.\n");
    #endif

	return SendDataEx(iClient, sData, strlen(sData));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the CriticalSection was not free,
	the client has data waiting to be sent, or if the client
	has been disconnected or is pending a disconnect.
*/
bool SocketServer::SetSendDataEx(int iClient, char *sData, int iSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SetSendDataEx.\n");
    #endif

    if(!IsClientDataPending(iClient))
    {
        if(!TryEnterCriticalSection(&CriticalSection1))
            return false;

        if(bcConnected[iClient] && !bcDisconnect[iClient])
        {
            memcpy(scSendBuf[iClient], sData, iSize);
            icSendBufSz[iClient] = iSize;
        }
        else return false;

        LeaveCriticalSection(&CriticalSection1);
        return true;
    }

	return false;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the CriticalSection was not free,
	the client has data waiting to be sent, or if the client
	has been disconnected or is pending a disconnect.
*/
bool SocketServer::SetSendData(int iClient, char *sData)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SetSendData.\n");
    #endif

	return SetSendDataEx(iClient, sData, strlen(sData));
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
*/
bool SocketServer::SetNextSendDataEx(int iClient, char *sData, int iSize)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SetNextSendDataEx.\n");
    #endif

    bool bRetVal = false;

    while(!(bRetVal = SetSendDataEx(iClient, sData, iSize)) && bcConnected[iClient] && !bcDisconnect[iClient])
    {
        Sleep(1);
    }

	return bRetVal;
}

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
/*
	This function is used to set a clients send buffer.

	Returns TRUE if the data was set successfully.
	Returns FALSE if the client has been disconnected or is pending a disconnect.
*/
bool SocketServer::SetNextSendData(int iClient, char *sData)
{
    #ifdef _DEBUG_SOCK_SERVER
    printf("DEBUG: [In Proc]> SocketServer::SetNextSendData.\n");
    #endif

	return SetNextSendDataEx(iClient, sData, strlen(sData));
}

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
